unit Grafix;

(*********************************************
Custom graphics classes and drawing routines.
*********************************************)

interface

uses
  Windows, SysUtils, Classes;

const
  MAX_VERTEX = 40;

type

  EGrafix = class( Exception );
  TDIBOrientation = ( orTopDown, orBottomUp );
  TQuadAngle = ( qa0, qa90, qa180, qa270 );

  TPolygon = class( TObject )
  private
    FClosed: boolean;
  protected
    function GetNum: integer;
    function GetPoint( n: integer ): TPoint;
    procedure SetPoint( n: integer; pt: TPoint );
  public
    pts: array[0..MAX_VERTEX - 1] of TPoint;
    NumVertex: integer;
    procedure Assign( p: TPolygon );
    procedure AddPoint( pt: TPoint );
    procedure Rotate( Degrees: integer );
    property Closed: boolean read FClosed write FClosed;
  end;

(***************************************************
Graphics routines
***************************************************)
function ClipLine( var x1, y1, x2, y2: integer; const rect: TRect ): boolean;
procedure CopyBlock( pSrc, pDest: pointer; nSrcWidth, nSrcHeight: integer;
  nSrcBlockWidth, nSrcBlockHeight: integer; nSrcX, nSrcY: integer; nDestWidth,
  nDestHeight: integer; nDestX, nDestY: integer; OrSrc, OrDest: TDIBOrientation );
procedure CopyBlockTrans( pSrc, pDest: pointer; nSrcWidth, nSrcHeight: integer;
  nSrcBlockWidth, nSrcBlockHeight: integer; nSrcX, nSrcY: integer; nDestWidth,
  nDestHeight: integer; nDestX, nDestY: integer; nTransColor: byte; OrSrc,
  OrDest: TDIBOrientation );
procedure CopyBlockTransQuad( pSrc, pDest: pointer; nSrcWidth, nSrcHeight: integer;
  nSrcBlockWidth, nSrcBlockHeight: integer; nSrcX, nSrcY: integer; nDestWidth,
  nDestHeight: integer; nDestX, nDestY: integer; nTransColor: byte; OrSrc,
  OrDest: TDIBOrientation; QuadAngle: TQuadAngle );
procedure DrawCircle( pDest: pointer; x, y, nRadius, nDestWidth, nDestHeight: integer;
  nColor: byte );
procedure DrawDIB( pSrc, pDest: pointer; nSrcWidth, nSrcHeight, nDestWidth,
  nDestX, nDestY: integer; nTrans: byte; nOrientation: byte ); stdcall; assembler;
procedure DrawLine( p: pointer; x1, y1, x2, y2, nDestWidth: integer; nColor: byte ); stdcall; assembler;
procedure DumpDIB( pSrc, pDest: pointer; nSrcWidth, nSrcHeight: integer;
  nOrientation: byte ); stdcall; assembler;
procedure FillBlock( pDest: pointer; nBlockWidth, nBlockHeight,
  nDestWidth, nDestHeight, nDestX, nDestY: integer;
  nColor: byte; OrDest: TDIBOrientation );
procedure FillMem( p: pointer; nBytes: integer; nVal: byte ); stdcall; assembler;
function PointInPolygon( poly: TPolygon; pt: TPoint ): boolean;
function IntToDegree( n: integer ): integer;

(***************************************************
The following arrays are lookup tables of common
trig functions.
***************************************************)
var
  lkupSin: array[0..360] of single;
  lkupCos: array[0..360] of single;
  lkupArcTan: array[0..360] of single;

implementation

var
  k: integer;

(*********************************************
Clip a line within a TRect.  Returns TRUE if
the clipped line can be drawn.
*********************************************)
function ClipLine( var x1, y1, x2, y2: integer; const rect: TRect ): boolean;
var
  bPoint1, bPoint2: boolean;
  yi, xi: integer;
  bRight, bLeft, bTop, bBottom: boolean;
  dx, dy: single;
begin
  Result := TRUE;
{ Test whether the line is completely visible }
  bPoint1 := ( ( x1 >= rect.Left ) and ( x1 <= rect.Right ) and ( y1 >= rect.Top ) and ( y1 <= rect.Bottom ) );
  bPoint2 := ( ( x2 >= rect.Left ) and ( x2 <= rect.Right ) and ( y2 >= rect.Top ) and ( y2 <= rect.Bottom ) );
{ Test endpoints }
  if bPoint1 and bPoint2 then
    Exit;
{ Test whether the line is completely invisible }
  if not bPoint1 and not bPoint2 then
  begin
    if ( ( ( x1 < rect.Left ) and ( x2 < rect.Left ) ) or
      ( ( x1 > rect.Right ) and ( x2 > rect.Right ) ) or
      ( ( y1 < rect.Top ) and ( y2 < rect.Top ) ) or
      ( ( y1 > rect.Bottom ) and ( y2 > rect.Bottom ) ) ) then
      begin
        Result := FALSE;
        Exit;
      end;
  end;
{ Reset the edge flags }
  bRight := FALSE;
  bLeft := FALSE;
  bTop := FALSE;
  bBottom := FALSE;
{ Is point 1 alone visible? }
  if bPoint1 or ( not bPoint1 and not bPoint2 ) then
  begin
    dx := x2 - x1;
    dy := y2 - y1;
    if x2 > rect.Right then
    begin
      bRight := TRUE;
      if dx <> 0 then
        yi := Trunc( 0.5 + ( dy / dx ) * ( rect.Right - x1 ) + y1 )
      else
        yi := -1;
    end
    else if x2 < rect.Left then
    begin
      bLeft := TRUE;
      if dx <> 0 then
        yi := Trunc( 0.5 + ( dy / dx ) * ( rect.Left - x1 ) + y1 )
      else
        yi := -1;
    end;
    if y2 > rect.Bottom then
    begin
      bBottom := TRUE;
      if dy <> 0 then
        xi := Trunc( 0.5 + ( dx / dy ) * ( rect.Bottom - y1 ) + x1 )
      else
        xi := -1;
    end
    else if y2 < rect.Top then
    begin
      bTop := TRUE;
      if dy <> 0 then
        xi := Trunc( 0.5 + ( dx / dy ) * ( rect.Top - y1 ) + x1 )
      else
        xi := -1;
    end;
    if bRight and ( ( yi >= rect.Top ) and ( yi <= rect.Bottom ) ) then
    begin
      x2 := rect.Right;
      y2 := yi;
    end
    else if bLeft and ( ( yi >= rect.Top ) and ( yi <= rect.Bottom ) ) then
    begin
      x2 := rect.Left;
      y2 := yi;
    end;
    if bBottom and ( ( xi >= rect.Left ) and ( xi <= rect.Right ) ) then
    begin
      x2 := xi;
      y2 := rect.Bottom;
    end
    else if bTop and ( ( xi >= rect.Left ) and ( xi <= rect.Right ) ) then
    begin
      x2 := xi;
      y2 := rect.Top;
    end;
  end;
{ Reset the edge flags }
  bRight := FALSE;
  bLeft := FALSE;
  bTop := FALSE;
  bBottom := FALSE;
{ Test the second endpoint }
  if bPoint2 or ( not bPoint1 and not bPoint2 ) then
  begin
    dx := x1 - x2;
    dy := y1 - y2;
    if x1 > rect.Right then
    begin
      bRight := TRUE;
      if dx <> 0 then
        yi := Trunc( 0.5 + ( dy / dx ) * ( rect.Right - x2 ) + y2 )
      else
        yi := -1;
    end
    else if x1 < rect.Left then
    begin
      bLeft := TRUE;
      if dx <> 0 then
        yi := Trunc( 0.5 + ( dy / dx ) * ( rect.Left - x2 ) + y2 )
      else
        yi := -1;
    end;
    if y1 > rect.Bottom then
    begin
      bBottom := TRUE;
      if dy <> 0 then
        xi := Trunc( 0.5 + ( dx / dy ) * ( rect.Bottom - y2 ) + x2 )
      else
        xi := -1;
    end
    else if y1 < rect.Top then
    begin
      bTop := TRUE;
      if dy <> 0 then
        xi := Trunc( 0.5 + ( dx / dy ) * ( rect.Top - y2 ) + x2 )
      else
        xi := -1;
    end;
    if bRight and ( ( yi >= rect.Top ) and ( yi <= rect.Bottom ) ) then
    begin
      x1 := rect.Right;
      y1 := yi;
    end
    else if bLeft and ( ( yi >= rect.Top ) and ( yi <= rect.Bottom ) ) then
    begin
      x1 := rect.Left;
      y1 := yi;
    end;
    if bBottom and ( ( xi >= rect.Left ) and ( xi <= rect.Right ) ) then
    begin
      x1 := xi;
      y1 := rect.Bottom;
    end
    else if bTop and ( ( xi >= rect.Left ) and ( xi <= rect.Right ) ) then
    begin
      x1 := xi;
      y1 := rect.Top;
    end;
  end;
  Result := ( x1 >= rect.Left ) and ( x1 <= rect.Right ) and
     ( x2 >= rect.Left ) and ( x2 <= rect.Right ) and
     ( y1 >= rect.Top ) and ( y2 <= rect.Bottom ) and
     ( y2 >= rect.Top ) and ( y2 <= rect.Bottom );
end;

procedure FillBlock( pDest: pointer; nBlockWidth, nBlockHeight,
  nDestWidth, nDestHeight, nDestX, nDestY: integer;
  nColor: byte; OrDest: TDIBOrientation );
var
  rectDest, rectBlock, rectIntersect: TRect;
  nOffDest, nIncDest: integer;
begin
  rectDest := Rect( 0, 0, nDestWidth - 1, nDestHeight - 1 );
  rectBlock := Rect( nDestX, nDestY, nDestX + nBlockWidth - 1, nDestY + nBlockHeight - 1 );
  if not IntersectRect( rectIntersect, rectDest, rectBlock ) then
    Exit;
{ Check for clipping and modify the destination/source rectangles accordingly }
  if nDestX < 0 then
  begin
    Dec( nBlockWidth, Abs( nDestX ) );
    nDestX := 0;
  end;
  if ( nDestX + nBlockWidth - 1 ) >= nDestWidth then
    Dec( nBlockWidth, ( ( nDestX + nBlockWidth ) - nDestWidth ) );
  if nDestY < 0 then
  begin
    Dec( nBlockHeight, Abs( nDestY ) );
    nDestY := 0;
  end;
  if ( nDestY + nBlockHeight - 1 ) >= nDestHeight then
     Dec( nBlockHeight, ( ( nDestY + nBlockHeight ) - nDestHeight ) );
{ Fill the block }
  if orDest = orBottomUp then
  begin
    nOffDest := nDestWidth * ( nDestHeight - nDestY - 1 ) + nDestX;
    nIncDest := -( nDestWidth + nBlockWidth );
  end
  else
  begin
    nOffDest := nDestWidth * nDestY + nDestX;
    nIncDest := nDestWidth - nBlockWidth;
  end;
asm
  push  esi
  push  edi
  push  ebx
{ Set destination index }
  mov   edi,[pDest]
  add   edi,[nOffDest]
{ How many rows to copy? }
  mov   ebx,[nBlockHeight]
{ Begin copying row data }
  mov   al,[nColor]
@OuterLoop:
  mov   ecx,[nBlockWidth]
  rep stosb
{ After each row, bump the pointers }
  add   edi,[nIncDest]
  dec   ebx
  jne   @OuterLoop
  pop ebx
  pop edi
  pop esi
end;
end;

(***************************************************
Copy a rectangular area from source to destination
memory, perform clipping.
***************************************************)
procedure CopyBlock( pSrc, pDest: pointer;
  nSrcWidth, nSrcHeight: integer;
  nSrcBlockWidth, nSrcBlockHeight: integer;
  nSrcX, nSrcY: integer;
  nDestWidth, nDestHeight: integer;
  nDestX, nDestY: integer;
  OrSrc, OrDest: TDIBOrientation );
var
  rectDest, rectDestBlock, rectIntersect: TRect;
  nOffSrc, nOffDest, nIncSrc, nIncDest: integer;
begin
{ First see if the destination area is completely outside of the destination block }
  rectDest := Rect( 0, 0, nDestWidth - 1, nDestHeight - 1 );
  rectDestBlock := Rect( nDestX, nDestY, nDestX + nSrcBlockWidth - 1, nDestY + nSrcBlockHeight - 1 );
  if not IntersectRect( rectIntersect, rectDest, rectDestBlock ) then
    Exit;
{ Check for clipping and modify the destination/source rectangles accordingly }
  if nDestX < 0 then
  begin
    Inc( nSrcX, Abs( nDestX ) );
    Dec( nSrcBlockWidth, Abs( nDestX ) );
    nDestX := 0;
  end;
  if ( nDestX + nSrcBlockWidth - 1 ) >= nDestWidth then
     Dec( nSrcBlockWidth, ( ( nDestX + nSrcBlockWidth ) - nDestWidth ) );
  if nDestY < 0 then
  begin
    Dec( nSrcBlockHeight, Abs( nDestY ) );
    Inc( nSrcY, Abs( nDestY ) );
    nDestY := 0;
  end;
  if ( nDestY + nSrcBlockHeight - 1 ) >= nDestHeight then
     Dec( nSrcBlockHeight, ( ( nDestY + nSrcBlockHeight ) - nDestHeight ) );
{ Copy the block }
  if orSrc = orBottomUp then
  begin
    nOffSrc := nSrcWidth * ( nSrcHeight - nSrcY - 1 ) + nSrcX;
    nIncSrc := -( nSrcWidth + nSrcBlockWidth );
  end
  else
  begin
    nOffSrc := nSrcWidth * nSrcY + nSrcX;
    nIncSrc := nSrcWidth - nSrcBlockWidth;
  end;
  if orDest = orBottomUp then
  begin
    nOffDest := nDestWidth * ( nDestHeight - nDestY - 1 ) + nDestX;
    nIncDest := -( nDestWidth + nSrcBlockWidth );
  end
  else
  begin
    nOffDest := nDestWidth * nDestY + nDestX;
    nIncDest := nDestWidth - nSrcBlockWidth;
  end;
asm
  push  esi
  push  edi
  push  ebx
{ Set source/destination indicies }
  mov  eax, ds
  mov  es, eax
  mov  esi,[pSrc]
  add  esi,[nOffSrc]
  mov  edi,[pDest]
  add  edi,[nOffDest]
{ How many rows to copy? }
  mov  edx,[nSrcBlockHeight]
  mov  eax,[nSrcBlockWidth]
  mov  ebx, eax
  shr  ebx,2
  and  eax,$03
{ Begin copying row data }
@OuterLoop:
  mov  ecx,ebx
  rep  movsd
  mov  ecx,eax
  rep  movsb
{ After each row, bump the pointers }
  add  esi,[nIncSrc]
  add  edi,[nIncDest]
  dec  edx
  jnz  @OuterLoop
  pop  ebx
  pop  edi
  pop  esi
end;
end;

(***************************************************
Copy a rectangular area from source to destination
memory, perform clipping, perform transparent check.
***************************************************)
procedure CopyBlockTrans( pSrc, pDest: pointer;
  nSrcWidth, nSrcHeight: integer;
  nSrcBlockWidth, nSrcBlockHeight: integer;
  nSrcX, nSrcY: integer;
  nDestWidth, nDestHeight: integer;
  nDestX, nDestY: integer;
  nTransColor: byte;
  OrSrc, OrDest: TDIBOrientation );
var
  rectDest, rectDestBlock, rectIntersect: TRect;
  nOffSrc, nOffDest, nIncSrc, nIncDest: integer;
begin
{ First see if the destination area is completely outside of the destination block }
  rectDest := Rect( 0, 0, nDestWidth - 1, nDestHeight - 1 );
  rectDestBlock := Rect( nDestX, nDestY, nDestX + nSrcBlockWidth - 1, nDestY + nSrcBlockHeight - 1 );
  if not IntersectRect( rectIntersect, rectDest, rectDestBlock ) then
    Exit;
{ Check for clipping and modify the destination/source rectangles accordingly }
  if nDestX < 0 then
  begin
    Inc( nSrcX, Abs( nDestX ) );
    Dec( nSrcBlockWidth, Abs( nDestX ) );
    nDestX := 0;
  end;
  if ( nDestX + nSrcBlockWidth - 1 ) >= nDestWidth then
     Dec( nSrcBlockWidth, ( ( nDestX + nSrcBlockWidth ) - nDestWidth ) );
  if nDestY < 0 then
  begin
    Dec( nSrcBlockHeight, Abs( nDestY ) );
    Inc( nSrcY, Abs( nDestY ) );
    nDestY := 0;
  end;
  if ( nDestY + nSrcBlockHeight - 1 ) >= nDestHeight then
     Dec( nSrcBlockHeight, ( ( nDestY + nSrcBlockHeight ) - nDestHeight ) );
{ Copy the block }
  if orSrc = orBottomUp then
  begin
    nOffSrc := nSrcWidth * ( nSrcHeight - nSrcY - 1 ) + nSrcX;
    nIncSrc := -( nSrcWidth + nSrcBlockWidth );
  end
  else
  begin
    nOffSrc := nSrcWidth * nSrcY + nSrcX;
    nIncSrc := nSrcWidth - nSrcBlockWidth;
  end;
  if orDest = orBottomUp then
  begin
    nOffDest := nDestWidth * ( nDestHeight - nDestY - 1 ) + nDestX;
    nIncDest := -( nDestWidth + nSrcBlockWidth );
  end
  else
  begin
    nOffDest := nDestWidth * nDestY + nDestX;
    nIncDest := nDestWidth - nSrcBlockWidth;
  end;
asm
  push esi
  push edi
  push ebx
{ Set source/destination indicies }
  mov  eax, ds
  mov  es, eax
  mov  esi,[pSrc]
  add  esi,[nOffSrc]
  mov  edi,[pDest]
  add  edi,[nOffDest]
{ How many rows to copy? }
  mov  ebx,[nSrcBlockHeight]
  mov  edx,[nSrcBlockWidth]
  mov  ah,[nTransColor]
{ Begin copying row data }
@OuterLoop:
  mov  ecx,edx
{ Check each byte against transparent color }
@InnerLoop:
  mov  al,ds:[esi]
  cmp  al,ah
  jz   @Trans
  mov  es:[edi],al
{ Increment source and dest pointers }
@Trans:
  inc  esi
  inc  edi
  loop @InnerLoop
{ After each row, bump the pointers }
  add  esi,[nIncSrc]
  add  edi,[nIncDest]
  dec  ebx
  jne  @OuterLoop
  pop  ebx
  pop  edi
  pop  esi
end;
end;

(*********************************************
Copy a transparent block, rotating by an angle within
90 degree increments.
*********************************************)
procedure CopyBlockTransQuad( pSrc, pDest: pointer; nSrcWidth, nSrcHeight: integer;
  nSrcBlockWidth, nSrcBlockHeight: integer; nSrcX, nSrcY: integer; nDestWidth,
  nDestHeight: integer; nDestX, nDestY: integer; nTransColor: byte; OrSrc,
  OrDest: TDIBOrientation; QuadAngle: TQuadAngle );
var
  rectDest, rectSource, rectUnion: TRect;
  nOffSrc, nIncSrc: integer;
  nOffDest, nIncInner, nIncOuter: integer;
begin
{ Destination buffer must be top down }
  if OrDest <> orTopDown then
    raise EGrafix.Create( 'Destination Buffer must be top down in CopyBlockTransQuad' );
{ This routine doesn't bother to clip the source rectangle }
  rectDest := Rect( 0, 0, nDestWidth - 1, nDestHeight - 1 );
  rectSource := Rect( nDestX, nDestY, nDestX + nSrcBlockWidth, nDestY + nSrcBlockHeight );
  UnionRect( rectUnion, rectDest, rectSource );
  if not EqualRect( rectUnion, rectDest ) then
    Exit;
{ Determine the offset and increment into the source buffer }
  if orSrc = orBottomUp then
  begin
    nOffSrc := nSrcWidth * ( nSrcHeight - nSrcY - 1 ) + nSrcX;
    nIncSrc := -( nSrcWidth + nSrcBlockWidth );
  end
  else
  begin
    nOffSrc := nSrcWidth * nSrcY + nSrcX;
    nIncSrc := nSrcWidth - nSrcBlockWidth;
  end;
{ Determine the offset and increment into the destination buffer
  !! Assume the destination buffer is top down !! }
  case QuadAngle of
    qa0:
    begin
      nOffDest := nDestWidth * nDestY + nDestX;
      nIncOuter := nDestWidth - nSrcBlockWidth;
      nIncInner := 1;
    end;
    qa90:
    begin
      nOffDest := nDestWidth * nDestY + nDestX + nSrcBlockWidth - 1;
      nIncOuter := -(nDestWidth * nSrcBlockHeight) - 1;
      nIncInner := nDestWidth;
    end;
    qa180:
    begin
      nOffDest := nDestWidth * nDestY + nDestX + nSrcBlockWidth - 1 + ( nDestWidth * (nSrcBlockHeight - 1) );
      nIncOuter := nSrcBlockWidth - nDestWidth;
      nIncInner := -1;
    end;
    qa270:
    begin
      nOffDest := nDestWidth * nDestY + nDestX + ( nDestWidth * (nSrcBlockHeight - 1) );
      nIncOuter := nDestWidth * nSrcBlockHeight + 1;
      nIncInner := -nDestWidth;
    end;
  end;
{ Perform the rendering }
asm
  push esi
  push edi
  push ebx
{ Set source/destination indicies }
  mov  eax, ds
  mov  es, eax
  mov  esi,[pSrc]
  add  esi,[nOffSrc]
  mov  edi,[pDest]
  add  edi,[nOffDest]
{ How many rows to copy? }
  mov  ebx,[nSrcBlockHeight]
  mov  edx,[nSrcBlockWidth]
  mov  ah,[nTransColor]
{ Begin copying row data }
@OuterLoop:
  mov  ecx,edx
{ Check each byte against transparent color }
@InnerLoop:
  mov  al,ds:[esi]
  cmp  al,ah
  jz   @Trans
  mov  es:[edi],al
{ Increment source and dest pointers }
@Trans:
  inc  esi
  add  edi,[nIncInner]
  loop @InnerLoop
{ After each row, bump the pointers }
  add  esi,[nIncSrc]
  add  edi,[nIncOuter]
  dec  ebx
  jne  @OuterLoop
  pop  ebx
  pop  edi
  pop  esi
end;
end;

(***************************************************
Implementation of Michener's circle algorithm.
***************************************************)
procedure DrawCircle( pDest: pointer; x, y, nRadius, nDestWidth, nDestHeight: integer;
  nColor: byte );
var
  x_, y_: integer;
  d: integer;
  pSurface: PByteArray;
  procedure SetPoint( xSet, ySet: integer );
  begin
     if ( xSet >= 0 ) and ( xSet < nDestWidth ) and ( ySet >= 0 ) and ( ySet < nDestHeight ) then
       pSurface[nDestWidth * ySet + xSet] := nColor;
  end;
begin
  pSurface := pDest;
  x_ := 0;
  y_ := nRadius;
  d := 3 - 2 * nRadius;
  while x_ <= y_ do
  begin
    SetPoint( x + x_, y + y_ );
    SetPoint( x - x_, y + y_ );
    SetPoint( x + x_, y - y_ );
    SetPoint( x - x_, y - y_ );
    SetPoint( x + y_, y + x_ );
    SetPoint( x - y_, y + x_ );
    SetPoint( x + y_, y - x_ );
    SetPoint( x - y_, y - x_ );
    if d < 0 then
      Inc( d, 4 * x_ + 6 )
    else
    begin
      Inc( d, 4 * ( x_ - y_ ) + 10 );
      Dec( y_ );
    end;
    Inc( x_ );
  end;
end;

(*********************************************
Draw a DIB ... handle transparent color ...
no clipping!
*********************************************)
procedure DrawDIB( pSrc, pDest: pointer; nSrcWidth, nSrcHeight, nDestWidth,
  nDestX, nDestY: integer; nTrans: byte; nOrientation: byte ); stdcall; assembler;
asm
  push  esi
  push  edi
  push  ebx
  mov   esi,[pSrc]
  mov   edi,[pDest]
  mov   al,[nOrientation]
  cmp   al,0
  jz    @TopDownDIB
@BottomUpDIB:
  mov   eax,[nDestWidth]
  mov   ebx,[nDestY]
  mul   ebx
  mov   ebx,[nDestX]
  add   eax,ebx
  add   edi,eax
  mov   ecx,[nSrcHeight]
@Outer:
  mov   ebx, [nSrcWidth]
@Inner:
  mov   al,ds:[esi]
  cmp   al,[nTrans]
  jz    @NoDraw
  mov   es:[edi], al
@NoDraw:
  inc   esi
  inc   edi
  dec   ebx
  jnz   @Inner
  mov   eax,edi
  mov   ebx,[nSrcWidth]
  sub   eax,ebx
  mov   ebx,[nDestWidth]
  add   eax,ebx
  mov   edi,eax
  dec   ecx
  jnz   @Outer
  jmp   @Done
@TopDownDIB:
  mov   eax,[nDestWidth]
  mov   ebx,[nSrcHeight]
  add   ebx,[nDestY]
  dec   ebx
  mul   ebx
  mov   ebx,[nDestX]
  add   eax,ebx
  add   edi,eax
  mov   ecx,[nSrcHeight]
@OuterTop:
  mov   ebx,[nSrcWidth]
@InnerTop:
  mov   al,ds:[esi]
  cmp   al,[nTrans]
  jz    @NoDrawTop
  mov   es:[edi],al
@NoDrawTop:
  inc   esi
  inc   edi
  dec   ebx
  jnz   @InnerTop
  mov   eax,edi
  mov   ebx,[nSrcWidth]
  sub   eax,ebx
  mov   ebx,[nDestWidth]
  sub   eax,ebx
  mov   edi,eax
  dec   ecx
  jnz   @OuterTop
@Done:
  pop ebx
  pop edi
  pop esi
end;

(*********************************************
Dump a DIB that has the exact same dimensions
as the buffer.
*********************************************)
procedure DumpDIB( pSrc, pDest: pointer; nSrcWidth, nSrcHeight: integer;
  nOrientation: byte ); stdcall; assembler;
asm
  push  esi
  push  edi
  push  ebx
  mov   esi, [pSrc]
  mov   edi, [pDest]
{ Store width / 2 in CX for number of words to move, push to stack }
  mov   edx, [nSrcWidth]
  shr   edx, 2
{ Check orientation of WinG DIB and process accordingly }
  mov   al, [nOrientation]
  cmp   al, 0
  jz    @TopDownDIB
{ Bottom up DIB is simply copying rows, first to last }
@BottomUpDIB:
  mov   ebx, nSrcHeight
@LoopBottomUp:
  mov   ecx, edx
  rep   movsd
  dec   ebx
  jnz   @LoopBottomUp
  jmp   @Done
{ Top down DIB copies copies first row of source to last of dest }
@TopDownDIB:
  mov   ebx, [nSrcHeight]
@LoopTopDown:
  mov   ecx, edx
{ Need to determine the correct destination offset to copy to }
  push  ebx
  mov   eax, ebx
  dec   eax
  mov   ebx, [nSrcWidth]
  push  edx
  mul   ebx
  pop   edx
  mov   edi, [pDest]
  add   edi, eax
  pop   ebx
  rep   movsd
  dec   ebx
  jnz   @LoopTopDown
@Done:
  pop  ebx
  pop  edi
  pop  esi
end;

(*********************************************
Implementation of Bresenham's line drawing
algorithm.  This procedure does not do bounds
checking, run the line through ClipLine before
calling this!
*********************************************)
procedure DrawLine( p: pointer; x1, y1, x2, y2, nDestWidth: integer; nColor: byte ); stdcall; assembler;
var
  nDirx, nDiry: byte;
asm
  push  esi
  push  edi
  push  ebx
{ The offset should begin at the first point }
  mov  esi, [p]
  mov  eax, [nDestWidth]       { Multiply destination width }
  mov  ebx, [y1]               { by Y pos of point 1 }
  mul  ebx
  mov  ebx, [x1]               { and add X pos of point 1 }
  add  eax, ebx
  add  esi, eax                { Store in ESI }
{ Store the error term in CX }
  xor  ecx, ecx
{ DeltaX in BX - DeltaY in DX }
  mov  ebx, [x2]
  sub  ebx, [x1]
  mov  edx, [y2]
  sub  edx, [y1]
{ Set directional variables ... 1=inc, 0=dec }
@TestDeltaX:
  cmp  ebx, 0
  jle  @X1
  mov  [nDirx], 1
  jmp  @TestDeltaY
@X1:
  mov  [nDirx], 0
  neg  ebx
@TestDeltaY:
  cmp  edx, 0
  jle  @Y1
  mov  [nDiry], 1
  jmp  @YDone
@Y1:
  mov  [nDiry], 0
  neg  edx
@YDone:
  mov  edi, 0
  mov  al, [nColor]
  cmp  ebx, edx
  jle  @Outer2
@Outer1:
  mov  es:[esi], al
  add  ecx, edx
  cmp  ecx, ebx
  jle  @NoProc1
  sub  ecx, ebx
  push eax
  mov  eax, [nDestWidth]
  cmp  [nDiry], 0
  jz   @Sub1
  add  esi, eax
  jmp  @Done1
@Sub1:
  sub  esi, eax
@Done1:
  pop  eax
@NoProc1:
  cmp  [nDirx], 0
  jz   @Sub2
  inc  esi
  jmp  @Done2
@Sub2:
  dec  esi
@Done2:
  inc  edi
  cmp  edi, ebx
  jle  @Outer1
  jmp  @Done
@Outer2:
  mov  es:[esi], al
  add  ecx, ebx
  cmp  ecx, 0
  jle  @NoProc2
  sub  ecx, edx
  cmp  [nDirx], 0
  jz   @Sub3
  inc  esi
  jmp  @NoProc2
@Sub3:
  dec  esi
@NoProc2:
  push eax
  mov  eax, [nDestWidth]
  cmp  [nDiry], 0
  jz   @Sub4
  add  esi, eax
  jmp  @Done4
@Sub4:
  sub  esi, eax
@Done4:
  pop  eax
  inc  edi
  cmp  edi, edx
  jle  @Outer2
@Done:
  pop  ebx
  pop  edi
  pop  esi
end;

(***************************************************
Fill a block of memory with the specified byte value.
***************************************************)
procedure FillMem( p: pointer; nBytes: integer; nVal: byte ); stdcall; assembler;
asm
  push edi
  push esi
  push ebx
  mov  edi,p
  mov  al,nVal
  mov  ah,al
  mov  bx,ax
  shl  eax,16
  mov  ax,bx
  mov  ebx,nBytes
  mov  ecx,ebx
  shr  ecx,2
  and  ebx,$03
  rep  stosd
  test ebx,ebx  // test & jump faster on a Pentium than rep stosb if ecx=0
  jz   @done
  mov  ecx,ebx
  rep  stosb
@done:
  pop  ebx
  pop  esi
  pop  edi
end;

(*********************************************
Test if a point is within a polygon
*********************************************)
function PointInPolygon( poly: TPolygon; pt: TPoint ): boolean;
var
  i, j: integer;
  pti, ptj: TPoint;
begin
  Result := FALSE;
  j := poly.NumVertex - 1;
  for i := 0 to poly.NumVertex - 1 do
  begin
    pti := poly.pts[i];
    ptj := poly.pts[j];
    if ( ( ( ( pti.y <= pt.y ) and ( pt.y < ptj.y ) ) or
       ( ( ptj.y <= pt.y ) and ( pt.y < pti.y ) ) ) and
       ( pt.x < ( ptj.x - pti.x ) * ( pt.y - pti.y ) / ( ptj.y - pti.y ) + pti.x ) ) then
      Result := not Result;
      j := i;
  end;
end;

function IntToDegree( n: integer ): integer;
begin
  while n < 0 do
    Inc( n, 360 );
  while n > 360 do
    Dec( n, 360 );
  Result := n;
end;

(***************************************************
TPolygon
***************************************************)
function TPolygon.GetNum: integer;
begin
  Result := NumVertex;
end;

function TPolygon.GetPoint( n: integer ): TPoint;
begin
  Result := pts[n];
end;

procedure TPolygon.setPoint( n: integer; pt: TPoint );
begin
  pts[n].X := pt.X;
  pts[n].Y := pt.Y;
end;

procedure TPolygon.AddPoint( pt: TPoint );
begin
  pts[NumVertex].X := pt.X;
  pts[NumVertex].Y := pt.Y;
  Inc( NumVertex );
end;

procedure TPolygon.Rotate( Degrees: integer );
var
  xSin, xCos: single;
  i: integer;
  x, y: integer;
begin
  xSin := lkupSin[Degrees];
  xCos := lkupCos[Degrees];
  for i := 0 to NumVertex - 1 do
  begin
    x := Round( pts[i].X * xCos - pts[i].Y * xSin );
    y := Round( pts[i].Y * xCos + pts[i].X * xSin );
    pts[i].X := x;
    pts[i].Y := y;
  end;
end;

procedure TPolygon.Assign( p: TPolygon );
var
  i: integer;
begin
  for i := 0 to p.NumVertex - 1 do
  begin
    pts[i].X := p.pts[i].X;
    pts[i].Y := p.pts[i].Y;
  end;
  NumVertex := p.NumVertex;
end;

(***************************************************
Set up the lookup tables on initialization
***************************************************)
initialization

for k := 0 to 360 do
begin
  lkupSin[k] := Cos( k * PI / 180.0 );
  lkupCos[k] := Sin( k * PI / 180.0 );
  lkupArcTan[k] := ArcTan( k * PI / 180.0 );
end;

end.
